;;; flp.el --- major mode for editing and running F-Logic

;; derived from prolog.el from GNU Emacs 19.28

;; Heinz Uphoff, uphoff@informatik.uni-freiburg.de, 20.1.95
;; Christian Schlepphorst, schlepph@informatik.uni-freiburg.de, 13.05.96
;; Changed: to new name FLORID 11.06.96

;;; Commentary:

;; This package provides a major mode for editing F-Logic.  It knows
;; about flp syntax and comments (well, sort of), and can send
;; regions, buffers, and files to an inferior F-Logic interpreter
;; process.

;;; Code:

(require 'comint)

;; This path has to be set at installation of the F-Logic-System!!!
(defvar flp-program-name "florid"
  "*Program name for invoking an inferior Flp with `run-flp'.")

(defvar flp-mode-syntax-table nil)
(defvar flp-mode-abbrev-table nil)
(defvar flp-mode-map nil)

(defvar flp-consult-string "sys.echo@(\"<Consulting region>\")[].\n sys.consult@(\"stdin\")[].\n"
  "*Consult stdin as plain input. ")

(defvar flp-forget-string "sys.forgetProgram[].\n sys.forgetIDB[].\n sys.echo@(\"<System reset>\")[].\n"
  "*reinitialise  system")

(defvar flp-offer-save t
  "*If non-nil, ask about saving modified buffers before 
\\[flp-consult-file] is run.")

(defvar flp-indent-width 4)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Do fontifying F-logic-syntax with the font-lock package (Xemacs 19.13)
;; If font-lock is not installed, there should be no problem

(make-face 'flp-font-lock-system-face)
(make-face 'flp-font-lock-arrow-face)
(copy-face 'default 'flp-font-lock-arrow-face)
(set-face-foreground 'flp-font-lock-arrow-face "red3")
(copy-face 'default 'flp-font-lock-system-face)
(set-face-foreground 'flp-font-lock-system-face "blue3")
(make-face-bold 'flp-font-lock-system-face)
(make-face-bold 'flp-font-lock-arrow-face)

(defconst flp-font-lock-keywords
   (list
    '("\\?-\\s-*\\(sys\\b\\(\\.\\w+\\)*\\)" 
      1 flp-font-lock-system-face)
    '("\\(\\?-\\|:-\\|\\.[ \t\n]\\)"
      1 flp-font-lock-arrow-face)
    '("\\(\\*?->>\\|\\*?->\\|\\*?=>>\\|\\*?=>\\)"
      1 flp-font-lock-system-face)
    '("\\(min\\|max\\|sum\\|avg\\|count\\)[ \t\n]*{" 
      1 flp-font-lock-system-face)
    '("\\(\\.\\|:\\|!\\|;\\)" 
      1 flp-font-lock-system-face)
    '("\\(\\[\\|\\]\\|{\\|}\\)"
      1 bold)
    '("\\b\\(address\\|error\\|float\\|parse\\|get\\|hrefs\\|integer\\|perl\\|match\\|pmatch\\|not\\|strcat\\|string\\|string2float\\|string2integer\\|string2object\\|strlen\\|substr\\|url\\|webdoc\\)\\b"
      1 flp-font-lock-system-face)
    )
  "Additional expressions to highlight in flp mode.")

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(if flp-mode-syntax-table
    ()
  (let ((table (make-syntax-table)))
    (modify-syntax-entry ?_ "w" table)
    (modify-syntax-entry ?\\ "\\" table)
    (modify-syntax-entry ?/ ". 1456" table)
    (modify-syntax-entry ?* ". 23" table)
    (modify-syntax-entry ?+ "." table)
    (modify-syntax-entry ?- "." table)
    (modify-syntax-entry ?= "." table)
    (modify-syntax-entry ?% "< b" table)
    (modify-syntax-entry ?\n "> b" table)
    (modify-syntax-entry ?< "." table)
    (modify-syntax-entry ?> "." table)
    (modify-syntax-entry ?\' "\"" table)
    (setq flp-mode-syntax-table table)))


(define-abbrev-table 'flp-mode-abbrev-table ())

(defun flp-mode-variables ()
  (set-syntax-table flp-mode-syntax-table)
  (setq local-abbrev-table flp-mode-abbrev-table)
  (make-local-variable 'paragraph-start)
  (setq paragraph-start (concat "^$\\|" page-delimiter)) ;;stolen from cplusplus 
  (make-local-variable 'paragraph-separate)
  (setq paragraph-separate paragraph-start)
  (make-local-variable 'paragraph-ignore-fill-prefix)
  (setq paragraph-ignore-fill-prefix t)
  (make-local-variable 'indent-line-function)
  (setq indent-line-function 'flp-indent-line)
  (make-local-variable 'comment-start)
  (setq comment-start "%")
  (make-local-variable 'comment-start-skip)
  (setq comment-start-skip "/\\*+ *\\|// *\\|% *") ;; stolen from cplusplus
  (make-local-variable 'comment-column)
  (setq comment-column 48)
  (make-local-variable 'comment-indent-function)
  (setq comment-indent-function 'flp-comment-indent)
;  (setq font-lock-keywords 'flp-font-lock-keywords)
)

(require 'browse-url) ;; allows to send url to a browser

(defun flp-mode-commands (map)
  (define-key map "\t" 'flp-indent-line)
  (define-key map "\C-c\C-l" 'flp-switch-to-flp-buffer)
  (define-key map "\C-c\C-b" 'flp-consult-buffer)
  (define-key map "\t" 'comint-dynamic-complete)
  (define-key map "\C-c\C-r" 'flp-consult-region)
  (define-key map "\C-c\C-c" 'flp-consult-first)
  (define-key map "\C-c\C-a" 'flp-consult-file)
  (define-key map "\C-c\C-s" 'flp-reset-system)
  (define-key map "\C-c\C-i" 'flp-interrupt)
  (define-key map "\C-\\" 'flp-break)
  (define-key map [(control button3)]  'browse-url-at-mouse)
)
(if flp-mode-map
    nil
  (setq flp-mode-map (make-sparse-keymap))
  (flp-mode-commands flp-mode-map))

;;;###autoload
(defun flp-mode ()
  "Major mode for editing F-Logic code.
Blank lines and `%%...' separate paragraphs.  `%'s start comments.
Commands:
\\{flp-mode-map}
Entry to this mode calls the value of `flp-mode-hook'
if that value is non-nil."
  (interactive)
  (kill-all-local-variables)
  (use-local-map flp-mode-map)
  (setq major-mode 'flp-mode)
  (setq mode-name "Flp")
  (flp-mode-variables)
  (setq comint-prompt-regexp "^?-")
  (run-hooks 'flp-mode-hook))

(defun flp-indent-line (&optional whole-exp)
  "Indent current line as Prolog code.
With argument, indent any additional lines of the same clause
rigidly along with this one (not yet)."
  (interactive "p")
  (let ((indent (flp-indent-level))
	(pos (- (point-max) (point))) beg)
    (beginning-of-line)
    (setq beg (point))
    (skip-chars-forward " \t")
    (if (zerop (- indent (current-column)))
	nil
      (delete-region beg (point))
      (indent-to indent))
    (if (> (- (point-max) pos) (point))
	(goto-char (- (point-max) pos)))
    ))

(defun flp-indent-level ()
  "Compute prolog indentation level."
  (save-excursion
    (beginning-of-line)
    (skip-chars-forward " \t")
    (cond
     ((looking-at "%%%") 0)		;Large comment starts
     ((looking-at "%[^%]") comment-column) ;Small comment starts
     ((bobp) 0)				;Beginning of buffer
     (t
      (let ((empty t) ind more less)
	(if (looking-at ")")
	    (setq less t)		;Find close
	  (setq less nil))
	;; See previous indentation
	(while empty
	  (forward-line -1)
	  (beginning-of-line)
 	  (if (bobp)
 	      (setq empty nil)
 	    (skip-chars-forward " \t")
 	    (if (not (or (looking-at "%[^%]") (looking-at "\n")))
 		(setq empty nil))))
 	(if (bobp)
 	    (setq ind 0)		;Beginning of buffer
	  (setq ind (current-column)))	;Beginning of clause
	;; See its beginning
	(if (looking-at "%%[^%]")
	    ind
	  ;; Real prolog code
	  (if (looking-at "(")
	      (setq more t)		;Find open
	    (setq more nil))
	  ;; See its tail
	  (end-of-flp-clause)
	  (or (bobp) (forward-char -1))
	  (cond ((looking-at "[,(;>]")
		 (if (and more (looking-at "[^,]"))
		     (+ ind flp-indent-width) ;More indentation
		   (max tab-width ind))) ;Same indentation
		((looking-at "-") tab-width) ;TAB
		((or less (looking-at "[^.]"))
		 (max (- ind flp-indent-width) 0)) ;Less indentation
		(t 0))			;No indentation
	  )))
     )))

(defun end-of-flp-clause ()
  "Go to end of clause in this line."
  (beginning-of-line 1)
  (let* ((eolpos (save-excursion (end-of-line) (point))))
    (if (re-search-forward comment-start-skip eolpos 'move)
	(goto-char (match-beginning 0)))
    (skip-chars-backward " \t")))

(defun flp-comment-indent ()
  "Compute prolog comment indentation."
  (cond ((looking-at "%%%") 0)
	((looking-at "%%") (flp-indent-level))
	(t
	 (save-excursion
	       (skip-chars-backward " \t")
	       ;; Insert one space at least, except at left margin.
	       (max (+ (current-column) (if (bolp) 0 1))
		    comment-column)))
	))


;;;
;;; Inferior flp mode
;;;
(defvar inferior-flp-mode-map nil)

(defun inferior-flp-mode ()
  "Major mode for interacting with an inferior flp process.

The following commands are available:
\\{inferior-flp-mode-map}

Entry to this mode calls the value of `flp-mode-hook' with no arguments,
if that value is non-nil.  Likewise with the value of `comint-mode-hook'.
`flp-mode-hook' is called after `comint-mode-hook'.

You can send text to the inferior flp from other buffers
using the commands `process-send-region', `process-send-string' and \\[flp-consult-region].

Commands:
Tab indents for Prolog; with argument, shifts rest
 of expression rigidly with the current line.
Paragraphs are separated only by blank lines and '%%'.
'%'s start comments.

Return at end of buffer sends line as input.
Return not at end copies rest of line to end and sends it.
\\[comint-kill-input] and \\[backward-kill-word] are kill commands, imitating normal Unix input editing.
\\[comint-interrupt-subjob] interrupts the shell or its current subjob if any.
\\[comint-stop-subjob] stops. \\[comint-quit-subjob] sends quit signal."
  (interactive)
  (require 'comint)
  (comint-mode)
  (setq major-mode 'inferior-flp-mode
	mode-name "Inferior-Flp"
	comint-prompt-regexp "^| [ ?][- ] *")
  (flp-mode-variables)
  (if inferior-flp-mode-map nil
    (setq inferior-flp-mode-map (copy-keymap comint-mode-map))
    (flp-mode-commands inferior-flp-mode-map))
  (use-local-map inferior-flp-mode-map)
  (run-hooks 'flp-mode-hook)
  (setq comint-input-ring-file-name (expand-file-name "~/.florid-history"))
  (comint-read-input-ring)
)

(defun run-flp-background ()
  "Run an inferior Flp process, input and output via buffer *flp*."
  (if (not (get-process "flp"))
      (save-excursion
	(set-buffer (make-comint "flp" flp-program-name))
	(inferior-flp-mode))))


;;;###autoload
(defun run-flp ()
  "Run an inferior Flp process, input and output via buffer *flp*, and
switch to the buffer."
  (interactive)
  (run-flp-background)
  (switch-to-buffer-other-window "*flp*")
)

(defun flp-consult-region (forget beg end)
  "Send the region to the flp process (made by \"M-x run-flp\" or created here)
If FORGET (prefix arg) is not nil, clear program and om before consulting.
PROBLEM: the region has to be correct, complete input."
  (interactive "P\nr")
  (run-flp-background)
  (save-excursion
    (if forget
	(flp-reset-system))
    (process-send-string "flp" flp-consult-string)
    (process-send-region "flp" beg end)
    (process-send-string "flp" "\n")
    (process-send-eof "flp")) ;Send eof to flp process.
  (display-buffer "*flp*"))

(defun flp-consult-region-as-query (forget beg end)
  "Send the region to the flp process as a query.
If FORGET (prefix arg) is not nil, clear program and om before consulting.
PROBLEM: every line of region has to be correct, complete input."
  (interactive "P\nr")
  (run-flp-background)
  (save-excursion
    (if forget
	(flp-reset-system))
    (process-send-region "flp" beg end)
    (process-send-string "flp" "\n"))		;May be unnecessary
  (display-buffer "*flp*"))

(defun flp-consult-buffer (forget)
  "Send the buffer region to the flp process like \\[flp-consult-region]."
  (interactive "P")
  (flp-consult-region forget (point-min-marker) (point-max-marker))
)

(defun flp-interrupt()
  (interactive)
  (interrupt-process "flp"))

(defun flp-break()
  (interactive)
  (quit-process "flp"))

(defun flp-reset-system ()
  (interactive)
  (run-flp-background)
  (process-send-string "flp" flp-forget-string)
  )

(defun flp-switch-to-flp-buffer ()
  (interactive)
  (run-flp-background)
  (pop-to-buffer "*flp*"))

(defun flp-consult-first ()
  "reset system before consulting buffer as file"
  (interactive)
  (flp-consult-file t))

(defun flp-consult-file (forget)
  "Prompt to save all buffers and run flp on current buffer's file.
If FORGET (prefix arg) is not nil, clear program and om before consulting.
This function is more useful than \\[flp-consult-buffer]."
  (interactive "P")
  (if (not (buffer-file-name))
      (error "Buffer does not seem to be associated with any file"))
  (if flp-offer-save
      (save-some-buffers))
  (run-flp-background)
  (if forget
      (flp-reset-system))
  (process-send-string "flp" (concat "sys.echo@(\"<Consulting "
			     buffer-file-name
			     ">\")[].\n"
		             "sys.consult@(\""
			     buffer-file-name
			     "\")[].\n"))
  (display-buffer "*flp*")
)

;;; flp.el ends here
















