;;; flx.el --- major mode for editing and running F-Logic

;; derived from prolog.el from GNU Emacs 19.28

;; Heinz Uphoff, uphoff@informatik.uni-freiburg.de, 20.1.95
;; Christian Schlepphorst, schlepph@informatik.uni-freiburg.de, 13.05.96
;; Changed: to new name FLORID 11.06.96

;;; Commentary:

;; This package provides a major mode for editing F-Logic.  It knows
;; about flx syntax and comments (well, sort of), and can send
;; regions, buffers, and files to an inferior F-Logic interpreter
;; process.

;;; Code:

(require 'comint)

;; This path has to be set at installation of the F-Logic-System!!!
(defvar flx-program-name "florid"
  "*Program name for invoking an inferior Flx with `run-flx'.")

(defvar flx-mode-syntax-table nil)
(defvar flx-mode-abbrev-table nil)
(defvar flx-mode-map nil)

(defvar flx-consult-string "sys.echo@(\"<Consulting region>\")[].\n sys.consult@(\"stdin\")[].\n"
  "*Consult stdin as plain input. ")

(defvar flx-forget-string "sys.forgetProgram[].\n sys.forgetIDB[].\n sys.echo@(\"<System reset>\")[].\n"
  "*reinitialise  system")

(defvar flx-offer-save t
  "*If non-nil, ask about saving modified buffers before 
\\[flx-consult-file] is run.")

(defvar flx-indent-width 4)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Do fontifying F-logic-syntax with the font-lock package (Xemacs 19.13)
;; If font-lock is not installed, there should be no problem

(make-face 'flx-font-lock-system-face)
(make-face 'flx-font-lock-arrow-face)
(copy-face 'default 'flx-font-lock-arrow-face)
(set-face-foreground 'flx-font-lock-arrow-face "red3")
(copy-face 'default 'flx-font-lock-system-face)
(set-face-foreground 'flx-font-lock-system-face "blue3")
(make-face-bold 'flx-font-lock-system-face)
(make-face-bold 'flx-font-lock-arrow-face)

(defconst flx-font-lock-keywords
   (list
    '("\\?-\\s-*\\(sys\\b\\(\\.\\w+\\)*\\)" 
      1 flx-font-lock-system-face)
    '("\\(\\?-\\|:-\\|\\.[ \t\n]\\)"
      1 flx-font-lock-arrow-face)
    '("\\(\\*?->>\\|\\*?->\\|\\*?=>>\\|\\*?=>\\)"
      1 flx-font-lock-system-face)
    '("\\(min\\|max\\|sum\\|avg\\|count\\)[ \t\n]*{" 
      1 flx-font-lock-system-face)
    '("\\(\\.\\|:\\|!\\|;\\)" 
      1 flx-font-lock-system-face)
    '("\\(\\[\\|\\]\\|{\\|}\\)"
      1 bold)
    '("\\b\\(address\\|error\\|float\\|parse\\|get\\|hrefs\\|integer\\|perl\\|match\\|pmatch\\|not\\|strcat\\|string\\|string2float\\|string2integer\\|string2object\\|strlen\\|substr\\|url\\|webdoc\\)\\b"
      1 flx-font-lock-system-face)
    )
  "Additional expressions to highlight in flx mode.")

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(if flx-mode-syntax-table
    ()
  (let ((table (make-syntax-table)))
    (modify-syntax-entry ?_ "w" table)
    (modify-syntax-entry ?\\ "\\" table)
    (modify-syntax-entry ?/ ". 14" table)
    (modify-syntax-entry ?* ". 23" table)
    (modify-syntax-entry ?+ "." table)
    (modify-syntax-entry ?- "." table)
    (modify-syntax-entry ?= "." table)
    (modify-syntax-entry ?% "< b" table)
    (modify-syntax-entry ?\n "> b" table)
    (modify-syntax-entry ?< "." table)
    (modify-syntax-entry ?> "." table)
    (modify-syntax-entry ?\' "\"" table)
    (setq flx-mode-syntax-table table)))


(define-abbrev-table 'flx-mode-abbrev-table ())

(defun flx-mode-variables ()
  (set-syntax-table flx-mode-syntax-table)
  (setq local-abbrev-table flx-mode-abbrev-table)
  (make-local-variable 'paragraph-start)
  (setq paragraph-start (concat "^$\\|" page-delimiter)) ;;stolen from cplusplus 
  (make-local-variable 'paragraph-separate)
  (setq paragraph-separate paragraph-start)
  (make-local-variable 'paragraph-ignore-fill-prefix)
  (setq paragraph-ignore-fill-prefix t)
  (make-local-variable 'indent-line-function)
  (setq indent-line-function 'flx-indent-line)
  (make-local-variable 'comment-start)
  (setq comment-start "%")
  (make-local-variable 'comment-start-skip)
  (setq comment-start-skip "/\\*+ *\\|// *\\|% *") ;; stolen from cplusplus
  (make-local-variable 'comment-column)
  (setq comment-column 48)
  (make-local-variable 'comment-indent-function)
  (setq comment-indent-function 'flx-comment-indent)
;  (setq font-lock-keywords 'flx-font-lock-keywords)
)

(require 'browse-url) ;; allows to send url to a browser

(defun flx-mode-commands (map)
  (define-key map "\t" 'flx-indent-line)
  (define-key map "\C-c\C-l" 'flx-switch-to-flx-buffer)
  (define-key map "\C-c\C-b" 'flx-consult-buffer)
  (define-key map "\t" 'comint-dynamic-complete)
  (define-key map "\C-c\C-r" 'flx-consult-region)
  (define-key map "\C-c\C-c" 'flx-consult-first)
  (define-key map "\C-c\C-a" 'flx-consult-file)
  (define-key map "\C-c\C-s" 'flx-reset-system)
  (define-key map "\C-c\C-i" 'flx-interrupt)
  (define-key map "\C-\\" 'flx-break)
  (define-key map [(control button3)]  'browse-url-at-mouse)
)
(if flx-mode-map
    nil
  (setq flx-mode-map (make-sparse-keymap))
  (flx-mode-commands flx-mode-map))

;;;###autoload
(defun flx-mode ()
  "Major mode for editing F-Logic code.
Blank lines and `%%...' separate paragraphs.  `%'s start comments.
Commands:
\\{flx-mode-map}
Entry to this mode calls the value of `flx-mode-hook'
if that value is non-nil."
  (interactive)
  (kill-all-local-variables)
  (use-local-map flx-mode-map)
  (setq major-mode 'flx-mode)
  (setq mode-name "Flx")
  (flx-mode-variables)
  (setq comint-prompt-regexp "^?-")
  (run-hooks 'flx-mode-hook))

(defun flx-indent-line (&optional whole-exp)
  "Indent current line as Prolog code.
With argument, indent any additional lines of the same clause
rigidly along with this one (not yet)."
  (interactive "p")
  (let ((indent (flx-indent-level))
	(pos (- (point-max) (point))) beg)
    (beginning-of-line)
    (setq beg (point))
    (skip-chars-forward " \t")
    (if (zerop (- indent (current-column)))
	nil
      (delete-region beg (point))
      (indent-to indent))
    (if (> (- (point-max) pos) (point))
	(goto-char (- (point-max) pos)))
    ))

(defun flx-indent-level ()
  "Compute prolog indentation level."
  (save-excursion
    (beginning-of-line)
    (skip-chars-forward " \t")
    (cond
     ((looking-at "%%%") 0)		;Large comment starts
     ((looking-at "%[^%]") comment-column) ;Small comment starts
     ((bobp) 0)				;Beginning of buffer
     (t
      (let ((empty t) ind more less)
	(if (looking-at ")")
	    (setq less t)		;Find close
	  (setq less nil))
	;; See previous indentation
	(while empty
	  (forward-line -1)
	  (beginning-of-line)
 	  (if (bobp)
 	      (setq empty nil)
 	    (skip-chars-forward " \t")
 	    (if (not (or (looking-at "%[^%]") (looking-at "\n")))
 		(setq empty nil))))
 	(if (bobp)
 	    (setq ind 0)		;Beginning of buffer
	  (setq ind (current-column)))	;Beginning of clause
	;; See its beginning
	(if (looking-at "%%[^%]")
	    ind
	  ;; Real prolog code
	  (if (looking-at "(")
	      (setq more t)		;Find open
	    (setq more nil))
	  ;; See its tail
	  (end-of-flx-clause)
	  (or (bobp) (forward-char -1))
	  (cond ((looking-at "[,(;>]")
		 (if (and more (looking-at "[^,]"))
		     (+ ind flx-indent-width) ;More indentation
		   (max tab-width ind))) ;Same indentation
		((looking-at "-") tab-width) ;TAB
		((or less (looking-at "[^.]"))
		 (max (- ind flx-indent-width) 0)) ;Less indentation
		(t 0))			;No indentation
	  )))
     )))

(defun end-of-flx-clause ()
  "Go to end of clause in this line."
  (beginning-of-line 1)
  (let* ((eolpos (save-excursion (end-of-line) (point))))
    (if (re-search-forward comment-start-skip eolpos 'move)
	(goto-char (match-beginning 0)))
    (skip-chars-backward " \t")))

(defun flx-comment-indent ()
  "Compute prolog comment indentation."
  (cond ((looking-at "%%%") 0)
	((looking-at "%%") (flx-indent-level))
	(t
	 (save-excursion
	       (skip-chars-backward " \t")
	       ;; Insert one space at least, except at left margin.
	       (max (+ (current-column) (if (bolp) 0 1))
		    comment-column)))
	))


;;;
;;; Inferior flx mode
;;;
(defvar inferior-flx-mode-map nil)

(defun inferior-flx-mode ()
  "Major mode for interacting with an inferior flx process.

The following commands are available:
\\{inferior-flx-mode-map}

Entry to this mode calls the value of `flx-mode-hook' with no arguments,
if that value is non-nil.  Likewise with the value of `comint-mode-hook'.
`flx-mode-hook' is called after `comint-mode-hook'.

You can send text to the inferior flx from other buffers
using the commands `process-send-region', `process-send-string' and \\[flx-consult-region].

Commands:
Tab indents for Prolog; with argument, shifts rest
 of expression rigidly with the current line.
Paragraphs are separated only by blank lines and '%%'.
'%'s start comments.

Return at end of buffer sends line as input.
Return not at end copies rest of line to end and sends it.
\\[comint-kill-input] and \\[backward-kill-word] are kill commands, imitating normal Unix input editing.
\\[comint-interrupt-subjob] interrupts the shell or its current subjob if any.
\\[comint-stop-subjob] stops. \\[comint-quit-subjob] sends quit signal."
  (interactive)
  (require 'comint)
  (comint-mode)
  (setq major-mode 'inferior-flx-mode
	mode-name "Inferior-Flx"
	comint-prompt-regexp "^| [ ?][- ] *")
  (flx-mode-variables)
  (if inferior-flx-mode-map nil
    (setq inferior-flx-mode-map (copy-keymap comint-mode-map))
    (flx-mode-commands inferior-flx-mode-map))
  (use-local-map inferior-flx-mode-map)
  (run-hooks 'flx-mode-hook)
  (setq comint-input-ring-file-name (expand-file-name "~/.florid-history"))
  (comint-read-input-ring)
)

(defun run-flx-background ()
  "Run an inferior Flx process, input and output via buffer *flx*."
  (if (not (get-process "flx"))
      (save-excursion
	(set-buffer (make-comint "flx" flx-program-name))
	(inferior-flx-mode))))


;;;###autoload
(defun run-flx ()
  "Run an inferior Flx process, input and output via buffer *flx*, and
switch to the buffer."
  (interactive)
  (run-flx-background)
  (switch-to-buffer-other-window "*flx*")
)

(defun flx-consult-region (forget beg end)
  "Send the region to the flx process (made by \"M-x run-flx\" or created here)
If FORGET (prefix arg) is not nil, clear program and om before consulting.
PROBLEM: the region has to be correct, complete input."
  (interactive "P\nr")
  (run-flx-background)
  (save-excursion
    (if forget
	(flx-reset-system))
    (process-send-string "flx" flx-consult-string)
    (process-send-region "flx" beg end)
    (process-send-string "flx" "\n")
    (process-send-eof "flx")) ;Send eof to flx process.
  (display-buffer "*flx*"))

(defun flx-consult-region-as-query (forget beg end)
  "Send the region to the flx process as a query.
If FORGET (prefix arg) is not nil, clear program and om before consulting.
PROBLEM: every line of region has to be correct, complete input."
  (interactive "P\nr")
  (run-flx-background)
  (save-excursion
    (if forget
	(flx-reset-system))
    (process-send-region "flx" beg end)
    (process-send-string "flx" "\n"))		;May be unnecessary
  (display-buffer "*flx*"))

(defun flx-consult-buffer (forget)
  "Send the buffer region to the flx process like \\[flx-consult-region]."
  (interactive "P")
  (flx-consult-region forget (point-min-marker) (point-max-marker))
)

(defun flx-interrupt()
  (interactive)
  (interrupt-process "flx"))

(defun flx-break()
  (interactive)
  (quit-process "flx"))

(defun flx-reset-system ()
  (interactive)
  (run-flx-background)
  (process-send-string "flx" flx-forget-string)
  )

(defun flx-switch-to-flx-buffer ()
  (interactive)
  (run-flx-background)
  (pop-to-buffer "*flx*"))

(defun flx-consult-first ()
  "reset system before consulting buffer as file"
  (interactive)
  (flx-consult-file t))

(defun flx-consult-file (forget)
  "Prompt to save all buffers and run flx on current buffer's file.
If FORGET (prefix arg) is not nil, clear program and om before consulting.
This function is more useful than \\[flx-consult-buffer]."
  (interactive "P")
  (if (not (buffer-file-name))
      (error "Buffer does not seem to be associated with any file"))
  (if flx-offer-save
      (save-some-buffers))
  (run-flx-background)
  (if forget
      (flx-reset-system))
  (process-send-string "flx" (concat 
                             "sys.echo@(\"Consulting "
			     buffer-file-name
			     "\")[].\n"
		             "sys.consult@(\""
			     buffer-file-name
			     "\")[].\n"))
  (display-buffer "*flx*")
)

;;; flx.el ends here
















