;;; lpx.el --- major mode for editing and running LoPiX

;; derived from prolog.el from GNU Emacs 19.28

;; Wolfgang May 1.1.2001

;;; Commentary:

;; This package provides a major mode for editing XPathLog.  It knows
;; about lopix syntax and comments (well, sort of), and can send
;; regions, buffers, and files to an inferior LoPiX interpreter
;; process.

;;; Code:

(require 'comint)

;; This path has to be set at installation of the LoPiX-System!!!
(defvar lpx-program-name "lopix"
  "*Program name for invoking an inferior LoPiX with `run-lpx'.")

(defvar lpx-mode-syntax-table nil)
(defvar lpx-mode-abbrev-table nil)
(defvar lpx-mode-map nil)

(defvar lpx-consult-string "sys.echo@(\"<Consulting region>\")[].\n sys.consult@(\"stdin\")[].\n"
  "*Consult stdin as plain input. ")

(defvar lpx-forget-string "sys.forgetProgram[].\n sys.forgetIDB[].\n sys.echo@(\"<System reset>\")[].\n"
  "*reinitialise  system")

(defvar lpx-offer-save t
  "*If non-nil, ask about saving modified buffers before 
\\[lpx-consult-file] is run.")

(defvar lpx-indent-width 4)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Do fontifying F-logic-syntax with the font-lock package (Xemacs 19.13)
;; If font-lock is not installed, there should be no problem

(make-face 'lpx-font-lock-system-face)
(make-face 'lpx-font-lock-arrow-face)
(copy-face 'default 'lpx-font-lock-arrow-face)
(set-face-foreground 'lpx-font-lock-arrow-face "red3")
(copy-face 'default 'lpx-font-lock-system-face)
(set-face-foreground 'lpx-font-lock-system-face "blue3")
(make-face-bold 'lpx-font-lock-system-face)
(make-face-bold 'lpx-font-lock-arrow-face)

(defconst lpx-font-lock-keywords
   (list
    '("\\?-\\s-*\\(sys\\b\\(\\.\\w+\\)*\\)" 
      1 lpx-font-lock-system-face)
    '("\\(\\?-\\|:-\\|\\.[ \t\n]\\)"
      1 lpx-font-lock-arrow-face)
    '("\\(\\*?->>\\|\\*?->\\|\\*?=>>\\|\\*?=>\\)"
      1 lpx-font-lock-system-face)
    '("\\(min\\|max\\|sum\\|avg\\|count\\)[ \t\n]*{" 
      1 lpx-font-lock-system-face)
    '("\\(\\.\\|:\\|!\\|;\\)" 
      1 lpx-font-lock-system-face)
    '("\\(\\[\\|\\]\\|{\\|}\\)"
      1 bold)
    '("\\b\\(address\\|error\\|float\\|parse\\|get\\|hrefs\\|integer\\|perl\\|match\\|pmatch\\|not\\|strcat\\|string\\|string2float\\|string2integer\\|string2object\\|strlen\\|substr\\|url\\|webdoc\\)\\b"
      1 lpx-font-lock-system-face)
    )
  "Additional expressions to highlight in lpx mode.")

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(if lpx-mode-syntax-table
    ()
  (let ((table (make-syntax-table)))
    (modify-syntax-entry ?_ "w" table)
    (modify-syntax-entry ?\\ "\\" table)
    (modify-syntax-entry ?/ ". 14" table)
    (modify-syntax-entry ?* ". 23" table)
    (modify-syntax-entry ?+ "." table)
    (modify-syntax-entry ?- "." table)
    (modify-syntax-entry ?= "." table)
    (modify-syntax-entry ?% "< b" table)
    (modify-syntax-entry ?\n "> b" table)
    (modify-syntax-entry ?< "." table)
    (modify-syntax-entry ?> "." table)
    (modify-syntax-entry ?\' "\"" table)
    (setq lpx-mode-syntax-table table)))


(define-abbrev-table 'lpx-mode-abbrev-table ())

(defun lpx-mode-variables ()
  (set-syntax-table lpx-mode-syntax-table)
  (setq local-abbrev-table lpx-mode-abbrev-table)
  (make-local-variable 'paragraph-start)
  (setq paragraph-start (concat "^$\\|" page-delimiter)) ;;stolen from cplusplus 
  (make-local-variable 'paragraph-separate)
  (setq paragraph-separate paragraph-start)
  (make-local-variable 'paragraph-ignore-fill-prefix)
  (setq paragraph-ignore-fill-prefix t)
  (make-local-variable 'indent-line-function)
  (setq indent-line-function 'lpx-indent-line)
  (make-local-variable 'comment-start)
  (setq comment-start "%")
;  (make-local-variable 'comment-start-skip)
;  (setq comment-start-skip "/\\*+ *\\|// *\\|% *") ;; stolen from cplusplus
  (make-local-variable 'comment-column)
  (setq comment-column 48)
  (make-local-variable 'comment-indent-function)
  (setq comment-indent-function 'lpx-comment-indent)
;  (setq font-lock-keywords 'lpx-font-lock-keywords)
)

(require 'browse-url) ;; allows to send url to a browser

(defun lpx-mode-commands (map)
  (define-key map "\t" 'lpx-indent-line)
  (define-key map "\C-c\C-l" 'lpx-switch-to-lpx-buffer)
  (define-key map "\C-c\C-b" 'lpx-consult-buffer)
  (define-key map "\t" 'comint-dynamic-complete)
  (define-key map "\C-c\C-r" 'lpx-consult-region)
  (define-key map "\C-c\C-c" 'lpx-consult-first)
  (define-key map "\C-c\C-a" 'lpx-consult-file)
  (define-key map "\C-c\C-s" 'lpx-reset-system)
  (define-key map "\C-c\C-i" 'lpx-interrupt)
  (define-key map "\C-\\" 'lpx-break)
  (define-key map [(control button3)]  'browse-url-at-mouse)
)
(if lpx-mode-map
    nil
  (setq lpx-mode-map (make-sparse-keymap))
  (lpx-mode-commands lpx-mode-map))

;;;###autoload
(defun lpx-mode ()
  "Major mode for editing XPathLog/LoPiX code.
Blank lines and `%%...' separate paragraphs.  `%'s start comments.
Commands:
\\{lpx-mode-map}
Entry to this mode calls the value of `lpx-mode-hook'
if that value is non-nil."
  (interactive)
  (kill-all-local-variables)
  (use-local-map lpx-mode-map)
  (setq major-mode 'lpx-mode)
  (setq mode-name "Lpx")
  (lpx-mode-variables)
  (setq comint-prompt-regexp "^?-")
  (run-hooks 'lpx-mode-hook))

(defun lpx-indent-line (&optional whole-exp)
  "Indent current line as Prolog code.
With argument, indent any additional lines of the same clause
rigidly along with this one (not yet)."
  (interactive "p")
  (let ((indent (lpx-indent-level))
	(pos (- (point-max) (point))) beg)
    (beginning-of-line)
    (setq beg (point))
    (skip-chars-forward " \t")
    (if (zerop (- indent (current-column)))
	nil
      (delete-region beg (point))
      (indent-to indent))
    (if (> (- (point-max) pos) (point))
	(goto-char (- (point-max) pos)))
    ))

(defun lpx-indent-level ()
  "Compute prolog indentation level."
  (save-excursion
    (beginning-of-line)
    (skip-chars-forward " \t")
    (cond
     ((looking-at "%%%") 0)		;Large comment starts
     ((looking-at "%[^%]") comment-column) ;Small comment starts
     ((bobp) 0)				;Beginning of buffer
     (t
      (let ((empty t) ind more less)
	(if (looking-at ")")
	    (setq less t)		;Find close
	  (setq less nil))
	;; See previous indentation
	(while empty
	  (forward-line -1)
	  (beginning-of-line)
 	  (if (bobp)
 	      (setq empty nil)
 	    (skip-chars-forward " \t")
 	    (if (not (or (looking-at "%[^%]") (looking-at "\n")))
 		(setq empty nil))))
 	(if (bobp)
 	    (setq ind 0)		;Beginning of buffer
	  (setq ind (current-column)))	;Beginning of clause
	;; See its beginning
	(if (looking-at "%%[^%]")
	    ind
	  ;; Real prolog code
	  (if (looking-at "(")
	      (setq more t)		;Find open
	    (setq more nil))
	  ;; See its tail
	  (end-of-lpx-clause)
	  (or (bobp) (forward-char -1))
	  (cond ((looking-at "[,(;>]")
		 (if (and more (looking-at "[^,]"))
		     (+ ind lpx-indent-width) ;More indentation
		   (max tab-width ind))) ;Same indentation
		((looking-at "-") tab-width) ;TAB
		((or less (looking-at "[^.]"))
		 (max (- ind lpx-indent-width) 0)) ;Less indentation
		(t 0))			;No indentation
	  )))
     )))

(defun end-of-lpx-clause ()
  "Go to end of clause in this line."
  (beginning-of-line 1)
  (let* ((eolpos (save-excursion (end-of-line) (point))))
    (if (re-search-forward comment-start-skip eolpos 'move)
	(goto-char (match-beginning 0)))
    (skip-chars-backward " \t")))

(defun lpx-comment-indent ()
  "Compute prolog comment indentation."
  (cond ((looking-at "%%%") 0)
	((looking-at "%%") (lpx-indent-level))
	(t
	 (save-excursion
	       (skip-chars-backward " \t")
	       ;; Insert one space at least, except at left margin.
	       (max (+ (current-column) (if (bolp) 0 1))
		    comment-column)))
	))


;;;
;;; Inferior lpx mode
;;;
(defvar inferior-lpx-mode-map nil)

(defun inferior-lpx-mode ()
  "Major mode for interacting with an inferior lpx process.

The following commands are available:
\\{inferior-lpx-mode-map}

Entry to this mode calls the value of `lpx-mode-hook' with no arguments,
if that value is non-nil.  Likewise with the value of `comint-mode-hook'.
`lpx-mode-hook' is called after `comint-mode-hook'.

You can send text to the inferior lpx from other buffers
using the commands `process-send-region', `process-send-string' and \\[lpx-consult-region].

Commands:
Tab indents for Prolog; with argument, shifts rest
 of expression rigidly with the current line.
Paragraphs are separated only by blank lines and '%%'.
'%'s start comments.

Return at end of buffer sends line as input.
Return not at end copies rest of line to end and sends it.
\\[comint-kill-input] and \\[backward-kill-word] are kill commands, imitating normal Unix input editing.
\\[comint-interrupt-subjob] interrupts the shell or its current subjob if any.
\\[comint-stop-subjob] stops. \\[comint-quit-subjob] sends quit signal."
  (interactive)
  (require 'comint)
  (comint-mode)
  (setq major-mode 'inferior-lpx-mode
	mode-name "Inferior-Lpx"
	comint-prompt-regexp "^| [ ?][- ] *")
  (lpx-mode-variables)
  (if inferior-lpx-mode-map nil
    (setq inferior-lpx-mode-map (copy-keymap comint-mode-map))
    (lpx-mode-commands inferior-lpx-mode-map))
  (use-local-map inferior-lpx-mode-map)
  (run-hooks 'lpx-mode-hook)
  (setq comint-input-ring-file-name (expand-file-name "~/.florid-history"))
  (comint-read-input-ring)
)

(defun run-lpx-background ()
  "Run an inferior Lpx process, input and output via buffer *lpx*."
  (if (not (get-process "lpx"))
      (save-excursion
	(set-buffer (make-comint "lpx" lpx-program-name))
	(inferior-lpx-mode))))


;;;###autoload
(defun run-lpx ()
  "Run an inferior Lpx process, input and output via buffer *lpx*, and
switch to the buffer."
  (interactive)
  (run-lpx-background)
  (switch-to-buffer-other-window "*lpx*")
)

(defun lpx-consult-region (forget beg end)
  "Send the region to the lpx process (made by \"M-x run-lpx\" or created here)
If FORGET (prefix arg) is not nil, clear program and om before consulting.
PROBLEM: the region has to be correct, complete input."
  (interactive "P\nr")
  (run-lpx-background)
  (save-excursion
    (if forget
	(lpx-reset-system))
    (process-send-string "lpx" lpx-consult-string)
    (process-send-region "lpx" beg end)
    (process-send-string "lpx" "\n")
    (process-send-eof "lpx")) ;Send eof to lpx process.
  (display-buffer "*lpx*"))

(defun lpx-consult-region-as-query (forget beg end)
  "Send the region to the lpx process as a query.
If FORGET (prefix arg) is not nil, clear program and om before consulting.
PROBLEM: every line of region has to be correct, complete input."
  (interactive "P\nr")
  (run-lpx-background)
  (save-excursion
    (if forget
	(lpx-reset-system))
    (process-send-region "lpx" beg end)
    (process-send-string "lpx" "\n"))		;May be unnecessary
  (display-buffer "*lpx*"))

(defun lpx-consult-buffer (forget)
  "Send the buffer region to the lpx process like \\[lpx-consult-region]."
  (interactive "P")
  (lpx-consult-region forget (point-min-marker) (point-max-marker))
)

(defun lpx-interrupt()
  (interactive)
  (interrupt-process "lpx"))

(defun lpx-break()
  (interactive)
  (quit-process "lpx"))

(defun lpx-reset-system ()
  (interactive)
  (run-lpx-background)
  (process-send-string "lpx" lpx-forget-string)
  )

(defun lpx-switch-to-lpx-buffer ()
  (interactive)
  (run-lpx-background)
  (pop-to-buffer "*lpx*"))

(defun lpx-consult-first ()
  "reset system before consulting buffer as file"
  (interactive)
  (lpx-consult-file t))

(defun lpx-consult-file (forget)
  "Prompt to save all buffers and run lpx on current buffer's file.
If FORGET (prefix arg) is not nil, clear program and om before consulting.
This function is more useful than \\[lpx-consult-buffer]."
  (interactive "P")
  (if (not (buffer-file-name))
      (error "Buffer does not seem to be associated with any file"))
  (if lpx-offer-save
      (save-some-buffers))
  (run-lpx-background)
  (if forget
      (lpx-reset-system))
  (process-send-string "lpx" (concat 
                             "sys.echo@(\"Consulting "
			     buffer-file-name
			     "\")[].\n"
		             "sys.consult@(\""
			     buffer-file-name
			     "\")[].\n"))
  (display-buffer "*lpx*")
)

;;; lpx.el ends here
















