/**
 * 
 * 
 * Franz Schenk <schenk@informatik.uni-goettingen.de>
 * 
 */
package org.semwebtech.util;

import java.util.Iterator;
import java.util.List;

import org.apache.log4j.Logger;

import com.hp.hpl.jena.rdf.model.InfModel;
import com.hp.hpl.jena.rdf.model.Model;
import com.hp.hpl.jena.rdf.model.ModelFactory;
import com.hp.hpl.jena.reasoner.rulesys.GenericRuleReasoner;
import com.hp.hpl.jena.reasoner.rulesys.Rule;


/** 
 * Build an InferenceModel with generic-rule-reasoning support.
 * - Load rules from file
 * - build InferenceModel with rules on top
 *   of model that is passed as a parameter
 * - return the inferenceModel, now enriched with
 *    a rulereasoner (applying either forward- or 
 *    backwardchaining rules)
 *    
 *  @author schenk
 */
public class JenaRules {

	private String rulesFileName ; 
	private Logger logger; 
	private String ruleEngineType = "fw";
	
	public JenaRules(){
		logger = Logger.getLogger("org.semwebtech.util.JenaRules");
	}
	
	/** 
	 * Create a RuleReasoner and build an InferenceModel.
	 * 
	 * @param inputRulesName
	 * @param ruleEngineType
	 * @param model
	 * @return InfModel
	 */
	public InfModel createRuleReasoner(
			String inputRulesName, 
			String ruleEngineType,
			Model model){
		if(setRuleEngineType(ruleEngineType))
			return createRuleReasoner(inputRulesName, model);
		else
			return null;
	}
	
	/**
	 * Create a RuleReasoner and build an InferenceModel.
	 * Use the default RuleEngine type (ForwardChaining)
	 *
	 * @param inputRulesName
	 * @param model
	 * @return InfModel
	 */
	public InfModel createRuleReasoner(String inputRulesName, Model model){
		this.rulesFileName = inputRulesName;
		InfModel infmodel;
		try {
			//---------------------------------------------------------------
			// Load the Rules
			List rules = Rule.rulesFromURL(rulesFileName);
			logger.debug("Rules loaded:");
			System.out.println("Rules loaded:");
			
			Iterator iter = rules.iterator();
			for ( ; iter.hasNext() ; ){
				Object a = iter.next();
				logger.debug("Rule  : "+a.toString());
				System.out.println("Rule  : "+a.toString());
			}
			//---------------------------------------------------------------
			
			//---------------------------------------------------------------
			// Build the RuleReasoner
			GenericRuleReasoner reasoner = new GenericRuleReasoner(rules);
			if (ruleEngineType.equals("bw"))
				reasoner.setMode(GenericRuleReasoner.BACKWARD);
			else 
				reasoner.setMode(GenericRuleReasoner.FORWARD_RETE);
			//---------------------------------------------------------------

			//---------------------------------------------------------------
			// Building an InfModel with the generic rule reasoner and rules
			infmodel = ModelFactory.createInfModel(reasoner, model);
			//---------------------------------------------------------------
			return infmodel;
		}
		catch(Exception e){
			System.out.println("... some error occured: " +e.toString());
			logger.error("Error building inference Model with rulereasoner: "+
					e.toString());
		}
 
		/* if we reach here it means, some error has occured. 
		 * Return null instead of a model */ 
		return null;
	}
	
	/**
	 * Set the RuleEngineType. Possible values are fw (forward chaining)
	 * or bw (backward-chaining).
	 * 
	 * @param type
	 * @return boolean (false in case of wrong arguments)
	 */
	public boolean setRuleEngineType(String type){
		if (type.toUpperCase().equals("BW") || type.toUpperCase().equals("FW") ){
			this.ruleEngineType = type;
			return true;
		}
		else{
			System.out.println("Allowed parameters are [bw|fw] for the type of the" +
					"ruleengine !");
			logger.debug("Allowed parameters are [bw|fw] for the type of the" +
					"ruleengine !");
			return false;
		}
	}
	
}