package org.semwebtech.util;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.URL;
import java.util.Properties;
import java.util.Vector;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.log4j.Logger;
import org.apache.log4j.PropertyConfigurator;
import org.mindswap.pellet.KnowledgeBase;
import org.mindswap.pellet.jena.OWLReasoner;
import org.mindswap.pellet.jena.PelletInfGraph;
import org.mindswap.pellet.jena.PelletReasonerFactory;

import com.hp.hpl.jena.ontology.Individual;
import com.hp.hpl.jena.ontology.OntClass;
import com.hp.hpl.jena.ontology.OntModel;
import com.hp.hpl.jena.ontology.OntModelSpec;
import com.hp.hpl.jena.query.Query;
import com.hp.hpl.jena.query.QueryException;
import com.hp.hpl.jena.query.QueryExecution;
import com.hp.hpl.jena.query.QueryExecutionFactory;
import com.hp.hpl.jena.query.QueryFactory;
import com.hp.hpl.jena.query.ResultSet;
import com.hp.hpl.jena.query.ResultSetFactory;
import com.hp.hpl.jena.query.ResultSetFormatter;
import com.hp.hpl.jena.rdf.model.InfModel;
import com.hp.hpl.jena.rdf.model.LiteralRequiredException;
import com.hp.hpl.jena.rdf.model.Model;
import com.hp.hpl.jena.rdf.model.ModelFactory;
import com.hp.hpl.jena.rdf.model.Resource;
import com.hp.hpl.jena.rdf.model.Statement;
import com.hp.hpl.jena.rdf.model.StmtIterator;
import com.hp.hpl.jena.reasoner.Reasoner;
import com.hp.hpl.jena.reasoner.ReasonerRegistry;
import com.hp.hpl.jena.reasoner.dig.DIGReasoner;
import com.hp.hpl.jena.reasoner.dig.DIGReasonerFactory;
import com.hp.hpl.jena.util.iterator.ExtendedIterator;
import com.hp.hpl.jena.vocabulary.ReasonerVocabulary;

/**
 * Tool for handling rdf-data. It is possible to 
 * - transform rdf-data from one language to another 
 * - use SPARQL-Queries on given rdf-data
 * - print a class-tree
 * 
 * @author Franz Schenk <schenk@informatik.uni-goettingen.de>
 */
public class ParseRDF {

	/**
	 * default Namespace that is used when none is given in the document or the
	 * query
	 */
	static final String defaultNS = "http://semwebtech.org/default-ont#";
	/** string array containing all accepted input languages */
	static final String[] knownInputLanguages = { "RDF/XML", "RDF/XML-ABBREV",
			"N-TRIPLE", "N3", "TURTLE" };
	/** string array containing all accepted output languages */
	static final String[] knownOutputLanguages = { "RDF/XML", "RDF/XML-ABBREV",
			"N-TRIPLE", "N3-PLAIN", "N3-PP", "N3-TRIPLE", "N3", "TURTLE" };
	/** string array containing all accepted result formats */
	static final String[] allowedOutputFormatters = { "PLAIN", "XML", "RDF" };
	/** default value for input language */
	static String inputLanguage = "";
	/** Default Value for output language */
	static String outputLanguage = "N3";
	/** default value for inputfilename */
	static String inputFileDefault = "<none>";
	/** default value for outputfilename */
	static String outputFile = "STDOUT";
	/** RuleFileName */
	static String ruleFile = "<none>";
	/** default value for queryfilename */
	static String queryFile = "test.sparql";
	/** default value for query support */
	static boolean inferenceSupport = false;
	/** default value for reasoner address */
	static String reasonerAddress = "INTERNAL";
	/** vector containing all inputfilenames */
	static Vector<String> inputFile = new Vector<String>();
	/** log4j Logger Object */
	static Logger logger;
	/** Query Result Output Format (aka pretty printing) */
	static String prettyPrint = "PLAIN";
	/** Package format gives a hint where to look for a log4j configuration */
	static String packageFormat = "JAR";
	/**
	 * String that is recognized as start of comment Note, that # is also used
	 * by the arq-module as a character introducing comments.
	 */
	static String commentString = "#";
	/** default value for output of classtree */
	static boolean classtree = false;

	public static void main(String[] args) {
		/*
		 * load a configuration file for log4j. The file is placed in the
		 * /org.semwebtech.util/ -directory of the jar.
		 */
		if (packageFormat.toUpperCase().equals("JAR"))
			PropertyConfigurator.configure(loadConfiguration());
		else if (packageFormat.toUpperCase().equals("TAR")) {
			/*
			 * If the program is delivered as a tarball containing all the other
			 * external libraries, its more convenient to load the
			 * log4j-configuration-file from the root of the package directory
			 */
			try {
				PropertyConfigurator.configure("log4j.properties");
			} catch (Exception e) {
				System.out.println("No log4j-configuration file found. Using"
						+ " the one provided with the package.");
			}
		}
		logger = Logger.getLogger("org.semwebtech.util.ParseRDF");
		logger.debug("### Processing CommandLine Arguments ");

		/*
		 * first parse the given parameters, test for the right number of
		 * arguments and switch to appropriate methods
		 */
		if (args.length == 0) {
			printHelp();
			System.exit(1);
		}

		int switchcount = 0;
		String modus = "unknown";

		for (int i = 0; i < args.length; i++) {
			if (args[i].startsWith("-")) {
				/* reset the switch */
				switchcount = 0;

				/* Read all the given commandline switches. */
				if (args[i].equals("-h")) {
					printHelp();
					System.exit(0);
				}
				if (args[i].equals("-command")) {
					System.out.println("Ceci n'est pas une commande.");
					System.exit(0);
				}
				if (args[i].equals("-if"))
					switchcount = 1;
				else if (args[i].equals("-of"))
					switchcount = 2;
				else if (args[i].equals("-il"))
					switchcount = 3;
				else if (args[i].equals("-ol"))
					switchcount = 4;
				else if (args[i].equals("-qf")) {
					modus = "q";
					switchcount = 5;
				} else if (args[i].equals("-rf"))
					switchcount = 8;
				else if (args[i].equals("-t"))
					modus = "t";
				else if (args[i].equals("-q"))
					modus = "q";
				else if (args[i].equals("-inf"))
					inferenceSupport = true;
				else if (args[i].equals("-r")) {
					inferenceSupport = true;
					switchcount = 6;
				} else if (args[i].equals("-pp"))
					switchcount = 7;
				else if (args[i].equals("-pellet")) {
					reasonerAddress = "PELLET";
					inferenceSupport = true;
				} else if (args[i].equals("-e")) {
					modus = "ct";
					classtree = true;
				} else {
					System.out.println("Unknown commandline-switch '" + args[i]
							+ "' .");
					printHelp();
					System.exit(1);
				}
			} else if (switchcount != 0) {
				/*
				 * enter here if a commandline switch has been detected that has
				 * to be used along with a value. Read that value here.
				 */
				if (switchcount == 1) /* Inputfilename */
					inputFile.addElement(parseAddress(args[i]));
				if (switchcount == 2) /* Outputfilename */
					outputFile = args[i];
				if (switchcount == 3) /* Inputlanguage */
					inputLanguage = args[i];
				if (switchcount == 4) /* Outputlanguage */
					outputLanguage = args[i];
				if (switchcount == 5) /* Queryfile */
					queryFile = args[i];
				if (switchcount == 6) /* Reasoneraddress */
					reasonerAddress = args[i];
				if (switchcount == 7) /* Query Result Output Format */
					prettyPrint = args[i];
				if (switchcount == 8) /* rulefilename */
					ruleFile = args[i];

				if (switchcount > 1)
					switchcount = 0;
			} else {
				System.out
						.println("You didn't specify what to do with this parameter:\n"
								+ args[i]
								+ "\nIf it is a filename, you have to give one of: [-qf | -if | -of ] ");
				System.exit(1);
			}
		}
		if (modus.equals("unknown")) {
			System.out.println("Use -t or -q to tell me what to do.");
			printHelp();
		} else {
			logger.debug("### Modus :<" + modus + ">");
			/*
			 * if no inputfilename is given, add the default name to the list of
			 * inputfilenames
			 */
			if (inputFile.size() == 0)
				inputFile.add(inputFileDefault);
			if (modus.equals("t"))
				translateDataFile();
			else if (modus.equals("q"))
				executeSPARQLQuery();
			else if (modus.equals("ct"))
				printClassTree();
		}
		System.out.println("Finished.");
	}

	/**
	 * Transform a given file containing rdf-data from one language to another
	 * First load a model, then pass it on to the output-function.
	 */
	static private void translateDataFile() {
		logger.debug("### Translate Data File");
		Model model = prepareModel();
		outputModel(model);
	}

	static private void outputModel(Model model) {
		/* check if the output-language is supported */
		outputLanguage = outputLanguage.toUpperCase();
		if (!checkLanguage(outputLanguage, knownOutputLanguages))
			System.out.println("Unknown Output-Language " + outputLanguage
					+ ". Known are:\n"
					+ arrayToString(knownOutputLanguages, " "));
		else {
			if (outputFile.equals("STDOUT")) {
				model.write(System.out, outputLanguage);
			} else {
				try {
					FileWriter writer = new FileWriter(outputFile);
					System.out.println("Writing Model in " + outputLanguage
							+ " ...\n");
					model.write(writer, outputLanguage);
				} catch (Exception e) {
					System.out.println("Error writing model to " + outputFile
							+ ":");
					System.out.println(e.toString());
				}
			}
			System.out.println("Finished output of Model.");

		}
	}

	/**
	 * uses a built-in function of the pellet knowledgebase-class, that gives
	 * the class tree of the actual graph (that the knowledgebase is equivalent
	 * to
	 */
	private static void printClassTree() {
		if (classtree) {
			if (!inferenceSupport
					&& !reasonerAddress.toUpperCase().equals("PELLET")) {
				System.out.println("No reasoner specified, using pellet as "
						+ "default for ClassTree ...");
				inferenceSupport = true;
				reasonerAddress = "PELLET";
			}
			Model model = prepareModel();
			System.out.println("\n------------- Class Tree ---------------");

			/* One way to do it */
			/*
			 * OWLReasoner owlReasoner = new OWLReasoner(); // The Pellet OWL
			 * Reasoner owlReasoner.load(model); // model: instance of
			 * owlReasoner.classify(); KnowledgeBase kb = owlReasoner.getKB();
			 */

			/* The other way to do it by building a KB */
			KnowledgeBase kb = ((PelletInfGraph) model.getGraph()).getKB();
			kb.realize();
			kb.printClassTree();
			System.out.println("----------------------------------------\n");
		}
	}

	/** check if a given language is supported as input or output language */
	private static boolean checkLanguage(String givenLanguage,
			String[] knownLanguages) {
		givenLanguage = givenLanguage.toUpperCase();
		boolean formatKnown = false;
		for (int i = 0; i < knownLanguages.length; i++)
			if (givenLanguage.equals(knownLanguages[i].toUpperCase()))
				formatKnown = true;
		return formatKnown;
	}

	/**
	 * Create a Model. Depending on the Parameters passed by the user either a
	 * model without inference support or a model backed by an internal /
	 * external reasoner is created.
	 * 
	 * @return Model
	 */
	private static Model prepareModel() {
		logger.debug("### Prepare the Model");

		/* load the first model from file */
		Model model = readFileIntoModel(inputFile.get(0));
		/*
		 * if we have more than just one model, we add the additional models to
		 * the first one
		 */
		if (inputFile.size() > 1) 
			for (int i = 1; i < inputFile.size(); i++) 
				model.add(readFileIntoModel(inputFile.get(i)));

		/*
		 * If inference-support is requested, a reasoner has to be configured
		 * and created
		 */
		if (inferenceSupport) {
			logger.debug("### Inference support");
			System.out.print("Building Model with inference support. ");

			/*
			 * if "internal" is given as the address of the reasoner, the
			 * internal inference engine, provided by jena, is used.
			 */
			if (reasonerAddress.toUpperCase().equals("INTERNAL")) {
				System.out.println("Using built-in reasoner.");
				Reasoner reasoner = ReasonerRegistry.getOWLReasoner();
				InfModel infmodel = ModelFactory
						.createInfModel(reasoner, model);
				return infmodel;
			} else if (reasonerAddress.toUpperCase().equals("PELLET")) {
				System.out.println(" Using Pellet-Plugin.");
				OntModel pelletmodel = ModelFactory
						.createOntologyModel(PelletReasonerFactory.THE_SPEC);
				pelletmodel.add(model);

				/*
				 * The reasoner has to be told of updates before it starts to
				 * update the inference model.
				 */
				pelletmodel.prepare();

				return pelletmodel;
			} else if (reasonerAddress.toUpperCase().equals("FW")
					|| reasonerAddress.toUpperCase().equals("BW")) {
				System.out.println("Using built in ruleengine ["
						+ reasonerAddress + "].");
				JenaRules jenarules = new JenaRules();
				InfModel infmodel = jenarules.createRuleReasoner(ruleFile,
						reasonerAddress, model);
				if (infmodel == null) {
					System.out
							.println("Couldn't Build the model with internal ruleengine support. Exit.");
					logger
							.debug("Couldn't build infmodel with ruleenginesupport.");
					System.exit(1);
					return null; // do we really reach here?
				} else
					return infmodel;
			}
			/* Otherwise we try to use the reasoner at the given address 
			 * (but do keep in mind: you really don't want to use a reasoner
			 *  via DIG interface !)*/
			else {
				try {
					System.out.println("Using reasoner at " + reasonerAddress
							+ ".");
					Resource conf = model.createResource();
					conf.addProperty(ReasonerVocabulary.EXT_REASONER_URL, model
							.createResource(reasonerAddress));
					DIGReasonerFactory drf = (DIGReasonerFactory) ReasonerRegistry
							.theRegistry().getFactory(DIGReasonerFactory.URI);
					DIGReasoner digReasoner = (DIGReasoner) drf.create(conf);
					OntModelSpec spec = new OntModelSpec(
							OntModelSpec.OWL_DL_MEM);
					spec.setReasoner(digReasoner);
					OntModel ontmodel = ModelFactory.createOntologyModel(spec,
							model);
					return ontmodel;
				} catch (Exception e) {
					System.out.println("Error: Failed building model.\n"
							+ e.toString());
					System.exit(1);
				}
				return null;
			}
		} else
			return model;
	}

	/**
	 * Load data from file/http into a rdf-Model
	 * 
	 * @param inputFileName
	 * @param inputLanguage
	 * @return Model
	 */
	private static Model readFileIntoModel(String inputFileName) {
		String language = inputLanguage;
		
		logger.debug("### Read Model from File");
		System.out.println("Reading Model from File " + inputFileName);
		Model model = ModelFactory.createDefaultModel();

		/* create a model, that will contain our rdf-data */
		try {
			/*
			 * if no inputLanguage is given: pre-read enough lines
			 * from the input dokument in order to guess the
			 * language */
			if (language.equals("")) {
				String filename = inputFileName;
				BufferedReader br;
				if (inputFileName.startsWith("file:")){
					/* remove the file:-protocol-spec */
					filename = inputFileName.substring("file:".length());
					br = new BufferedReader(new FileReader(filename));
					language = guessLanguage(br);
					br.close();
				}
				else { 
					/* the only other supported protocol is http !!*/
					if (! inputFileName.startsWith("http:")){
						System.out.println(
								"Protocol not supported. Cannot guess language.");
					}
					else{
						URL inputURL = new URL(inputFileName);
						br = new BufferedReader(
								new InputStreamReader(
										inputURL.openStream()));
						language = guessLanguage(br);
						br.close();
					}
				}
			}

			/* check if input-language is supported */
			language = language.toUpperCase();
			if (!checkLanguage(language, knownInputLanguages)) {
				System.out.println("Unknown Input-Language " + language
						+ ". Known are:\n"
						+ arrayToString(knownInputLanguages, " "));
				System.exit(1);
			}

			/* Read the model from the given address and the given language */
			model.read(inputFileName, language);
			
			/* reset the inputLanguage, so that for the next files it will be guessed
			 * as well */
		} catch (Exception e) {
			System.out.println("Error reading model from file.\n"
					+ e.toString() + "\n(Filename was: " + inputFileName);
			System.exit(1);
		}
		return model;
	}

	/**
	 * Load a given (text) File into a String (ignoring lines starting with #)
	 * Needed for SPARQL-queries with a FROM clause.
	 * 
	 * @param filename
	 * @return querystring
	 */
	private static String readFileIntoString(String filename) {
		logger.debug("### Read SPARQL-Query from File");

		/* Load a SPARQL query from file */
		String stringFromFile = new String();
		try {
			BufferedReader reader = new BufferedReader(new FileReader(new File(
					filename)));
			String line = reader.readLine();
			while (line != null) {
				if (!line.startsWith(commentString)) { /* ignore comments */
					stringFromFile += line + "\n";
				}
				line = reader.readLine();
			}
			reader.close();
		} catch (Exception e) {
			System.out.println(e.toString());
			stringFromFile = "empty";
			System.exit(1);
		}
		return stringFromFile;
	}

	/** Execute a SPARQL-Query */
	private static void executeSPARQLQuery() {
		/*
		 * load the query from file. (Note that although we read the query string
		 * here in order to do determine which kind of query we have and whether
		 * there are any inputfiles given in the query, the sparql-engine reads
		 * the query (for a second time) from file on its own.)
		 */
		String querystring = readFileIntoString(queryFile);

		/*
		 * in case the user wants inference support: all the datafiles that are
		 * given in the query-file via FROM-clause have to be loaded before in
		 * order to use a reasoner to infer additional statements
		 * 
		 * and now: always load the data into the model before querying. Hereby
		 * it is possible to give a filename in the FROM clause in the query and
		 * give another filename(s) at the commandline as well.
		 */
		//if (inferenceSupport)
		querystring = parseQuery(querystring);

		System.out.println("----------------------------------------");

		/*
		 * To match only the keyword ASK as an SPARQL-construct the comparison
		 * here looks for either '\nASK ' or '\n ASK '. Therefor, if you state a
		 * ASK-Query, the ask-keyword should start a new-line and must not be
		 * preceeded by any other character than a whitespace.
		 */
		Pattern regexPattern = Pattern.compile("\\n\\s*ASK\\s",
				Pattern.CASE_INSENSITIVE);
		Matcher regexMatcher = regexPattern.matcher(querystring);

		/* SPARQL ASK Query */
		if (!querystring.equals("empty") && regexMatcher.find()) {
			logger.debug("### Executing ASK-query");

			System.out.println("Executing SPARQL-ASK-query " + queryFile + "\n"
					+ querystring);
			QueryExecution qexec = null;
			try {
				// Query qu = QueryFactory.create(querystring, defaultNS);
				Query qu = QueryFactory.create(querystring);
				Model model = prepareModel();
				qexec = QueryExecutionFactory.create(qu, model);
				String resultString;
				if (qexec.execAsk())
					resultString = "Queryanswer: Yes.";
				else
					resultString = "Queryanswer: No.";
			}// Ende try
			catch (QueryException qe) {
				System.out.println("Fehler.");
			} finally {
				qexec.close();
			}
		}
		
		/* SPARQL construct Query */
		else if (!querystring.equals("empty")
				&& querystring.toUpperCase().contains("CONSTRUCT")) {
			logger.debug("### Executing CONSTRUCT-query");

			outputLanguage = outputLanguage.toUpperCase();
			if (!checkLanguage(outputLanguage, knownOutputLanguages)) {
				System.out.println("Unknown Output-Language " + outputLanguage
						+ ". Known are:\n"
						+ arrayToString(knownOutputLanguages, " "));
				System.exit(1);
			}
			System.out.println("Executing SPARQL-CONSTRUCT-query " + queryFile
					+ "\n" + querystring);
			QueryExecution qexec = null;
			try {
				// Query qu = QueryFactory.create(querystring, defaultNS);
				Query qu = QueryFactory.create(querystring);
				Model model = prepareModel();
				qexec = QueryExecutionFactory.create(qu, model);
				Model constrModel = qexec.execConstruct();

				if (outputFile.equals("STDOUT")) {
					constrModel.write(System.out, outputLanguage);
				} else {
					try {
						FileWriter writer = new FileWriter(outputFile);
						System.out.println("Writing constructed Model in "
								+ outputLanguage + " ...\n");
						constrModel.write(writer, outputLanguage);
					} catch (Exception e) {
						System.out.println("Error writing model to "
								+ outputFile + ":");
						System.out.println(e.toString());
					}
				}
				System.out
						.println("\n\nSuccessfully wrote constructed rdf-graph.");
			}// Ende try
			catch (QueryException qe) {
				System.out.println("Fehler.");
			} finally {
				qexec.close();
			}
		}
		
		/* SPARQL-Select query */
		else if (!querystring.equals("empty")) {
			logger.debug("### Executing SELECT-query");

			StringBuilder buffer = new StringBuilder();
			QueryExecution qexec = null;
			try {
				/* Print querystring */
				System.out.println("Executing SPARQL-query " + queryFile + "\n"
						+ querystring.replaceAll("\n\\s*\n", "\n"));
				
				/* now the sparql-engine reads the query from file */
				Query qu = QueryFactory.read(queryFile);

				Model model = prepareModel();

				/*
				 * build the knowledge base: though this is just a different
				 * representation of the rdf-graph, it is until now
				 * necessary in order to obtain the correct inference data
				 * from pellet (this was a bug in pellet with strange side
				 * effects; in other words: this is just a bloody
				 * work-around) KnowledgeBase kb =
				 * ((PelletInfGraph)model.getGraph()).getKB(); kb.realize();
				 */
				qexec = QueryExecutionFactory.create(qu, model);
				
				/* At this point we start the query */
				ResultSet results = qexec.execSelect();

				/*
				 * We want to give the variable bindings as well as the total
				 * number of results. In order to do so it is necessary to
				 * operate on a copy of the resultSet. Any attempt to work on
				 * the original resultset leads to an empty resultset after a
				 * single operation, be it the determination of its size or the
				 * output of the results.
				 */
				ResultSet copyOfResultSet = ResultSetFactory
						.copyResults(results);
				int numberOfResults = ResultSetFormatter
						.toList(copyOfResultSet).size();

				/* Now take a look at the results */
				if (formattedResultsOutput(copyOfResultSet))
					System.out.println("Successfully printed "
							+ numberOfResults + " query results.");
				qexec.close();
			} catch (Exception e) {
				System.out.println("Query failure: " + e.toString());
			}

		}

	}

	/**
	 * return a string composed of elements of given array, each delimited by
	 * given string
	 */
	private static String arrayToString(String[] stringarray, String delimiter) {
		String returnstring = "";
		for (int i = 0; i < stringarray.length - 1; i++)
			returnstring += stringarray[i] + delimiter;
		returnstring += stringarray[stringarray.length - 1] + " ";
		return returnstring;
	}

	/** add protocol 'file:' to address-string if no protocol is given. 
	 * */
	private static String parseAddress(String address) {
		if (address.contains(":"))
			return address;
		else
			return "file:" + address;
	}

	/**
	 * Helper function to parse a query, remove all FROM-clauses (but
	 * not in FROM NAMED) and add the URI's/Filenames that are given with
	 * the FROM clause to the list of input Filenames
	 * 
	 * @param queryString
	 * @return String
	 */
	private static String parseQuery(String queryString) {
		String regExpression = "FROM( )*<.*?>";

		Pattern regexPattern = Pattern.compile(regExpression,
				Pattern.CASE_INSENSITIVE);
		Matcher regexMatcher = regexPattern.matcher(queryString);
		boolean found = false;
		while (regexMatcher.find()) {
			if (!found) {
				if (inputFile.elementAt(0).toUpperCase().equals("<NONE>"))
					inputFile.remove(0); 
				found = true;
			}
			
			String match = regexMatcher.group();
			logger.debug("### parseQuery: Found match: " + match);
			int startpos = match.indexOf("<");
			int endpos = match.lastIndexOf(">");
			String fileNameToAdd = match.substring(startpos + 1, endpos);
			/*
			 * if the filename does not start with a protocol-specification
			 * (known are file: or http:) then add file:
			 */
			if (!fileNameToAdd.startsWith("file:")
					&& !fileNameToAdd.startsWith("http:"))
				fileNameToAdd = "file:" + fileNameToAdd;
			System.out
					.println("Loading File <"
							+ fileNameToAdd
							+ "> into model (FROM clause of the query)");
			inputFile.addElement(fileNameToAdd);
		}
		if (found)
			queryString = regexMatcher.replaceAll("");
		return queryString;
	}

	/**
	 * Load a log4j Properties File. The name of the File to be loaded is
	 * 'log4j.properties'. The location is within the same directory as this
	 * classfile. The intention is to be able to load a config-file inside the
	 * jar.
	 * 
	 * @return Properties
	 */
	public static Properties loadConfiguration() {
		Properties prop = new Properties();
		try {
			InputStream is = ParseRDF.class
					.getResourceAsStream("log4j.properties");
			prop.load(is);
			is.close();
		} catch (Exception e) {
			System.out.println(e.toString());
		}
		return prop;
	}

	/**
	 * Output of a SPARQL Query ResultSet. Default method is a plain text table
	 * containing variable bindings. Other possibilities are RDF, XML.
	 * 
	 * @param results
	 * @return boolean
	 */
	private static boolean formattedResultsOutput(ResultSet results) {

		/*
		 * make a copy of resultSet, otherwise the contents of the result set
		 * are forgotten when the program branches at the conditional (don't ask
		 * why)
		 */
		ResultSet copyOfResults = ResultSetFactory.copyResults(results);

		/* try some ResultSetFormatter */
		if (checkLanguage(prettyPrint, allowedOutputFormatters)) {
			System.out.print("Query result as ");
			if (prettyPrint.toUpperCase().equals("PLAIN")) {
				System.out.println(" plain text:");
				ResultSetFormatter.out(System.out, copyOfResults);
			}

			else if (prettyPrint.toUpperCase().equals("RDF")) {
				System.out.println(" RDF data, Outputlanguage <"
						+ outputLanguage + ">. ");
				if (!checkLanguage(outputLanguage, knownOutputLanguages)) {
					System.out.println("Unknown Output-Language "
							+ outputLanguage + ". Known are:\n"
							+ arrayToString(knownOutputLanguages, " "));
					return false;
				} else {
					ResultSetFormatter.outputAsRDF(System.out, outputLanguage
							.toUpperCase(), copyOfResults);
				}
			} else if (prettyPrint.toUpperCase().equals("XML")) {
				System.out.println(" XML:");
				ResultSetFormatter.outputAsXML(System.out, copyOfResults);
			}
		} else {
			System.out.println("Unknown result set format. Try one of { "
					+ arrayToString(allowedOutputFormatters, " | ") + " }");
			return false;
		}
		return true;
	}

	/** Print out a help message on how to use this program. */
	private static void printHelp() {
		System.out
				.println("Usage: java -jar org.semwebtech.util.jar -command\n"
						+ "Operating Modes:\n"
						+ "-h   print this helpmessage\n"
						+ "-t   translate data\n" + "-q   sparql-query\n"
						+ "-e   print the class tree (per default used together with "
						+ "the pellet inference engine)\n"
						+ "Additional parameters:\n" + "-if  inputfile(s) ["
						+ inputFileDefault
						+ "]\n"
						+ "     Examples: /path/to/test.n3\n"
						+ "               file:///path/to/test.n3\n"
						+ "               http://example.com/test.n3\n"
						+ "-of  outputfile ["
						+ outputFile
						+ "]\n"
						+ "-qf  file containing the SPARQL-Query ["
						+ queryFile
						+ "]\n"
						+ "     (if -qf is used -q can be ommitted)\n"
						+ "-rf  file containing the rules for the ruleengine ["
						+ ruleFile
						+ "]\n"
						+ "-il  inputlanguage ["
						+ inputLanguage
						+ "]"
						+ "   { "
						+ arrayToString(knownInputLanguages, " | ")
						+ "}\n"
						+ "     (if no language is specified the program tries to guess)\n"
						+ "-ol  outputlanguage ["
						+ outputLanguage
						+ "]"
						+ "  { "
						+ arrayToString(knownOutputLanguages, " | ")
						+ "}\n"
						+ "-inf turn on inference support ["
						+ inferenceSupport
						+ "]\n"
						+ "-r   reasoneraddress ["
						+ reasonerAddress
						+ "]"
						+ " { internal | pellet | fw | bw | http://someaddress:portnumber }\n"
						+ "     (-pellet is a shortcut for -r pellet)\n"
						+ "-pp  pretty print query results ["
						+ prettyPrint
						+ "]"
						+ "  { "
						+ arrayToString(allowedOutputFormatters, " | ") + "}");
	}

	/**
	 * Try to guess a model string's language
	 * 
	 * @author Jochen Kemnade , Franz Schenk
	 * 
	 * @param modelString
	 *            the model string
	 * @return the language
	 */
	public static String guessLanguage(String modelString) {
		int offset = 0;
		String language = null;
		while (offset < modelString.length()  
				&& Character.isWhitespace(modelString.charAt(offset))){ 
			offset++;
		}
		if (offset >= modelString.length())
			return null;
		if (modelString.startsWith("#"))
			return null;
		if (modelString.charAt(offset) == '<' && modelString.length()>=4 ) {
			if(modelString.substring(offset + 1, offset + 4)
					.toUpperCase().equals("!--")) {
				language = "RDF/XML";
			} else if (modelString.substring(offset + 1, offset + 5)
					.toUpperCase().equals("?XML")) {
				language = "RDF/XML";
			} else if (modelString.substring(offset + 1, offset + 8)
					.toUpperCase().equals("RDF:RDF")) {
				language = "RDF/XML"; 
			} else if (modelString.substring(offset + 1, offset + 9).toUpperCase()
					.equals("!DOCTYPE")) {
				language = "RDF/XML";
			} else {
				language = "N3";
			}
		} else {
			language = "N3";
		}
		return language;
	}
	
	/** 
	 * Try to guess what language the document that the stream points to
	 * is. Read from that stream until language could be determined or until
	 * end of stream.
	 *  
	 * @param br
	 */
	public static String guessLanguage(BufferedReader br){
		String language;
		try{
			String line = "";
			while ((line = br.readLine()) != null){
				if ((language = guessLanguage(line)) == null)
					continue;
				else{ 
					System.out.println("Assuming Language: "
							+ language);
					return language;
				}
			}
		}
		catch(Exception e){
			System.out.println("Couldn't read from address");
		}
		return null;
	}

}
